<?php defined('BASEPATH') OR exit('No direct script access allowed');
/**
* @package direct-as-a-service
* @subpackage controllers
* @filesource
*//** */
 
require_once 'mailbox_controller.php';

/**
* API methods to manage a mailbox's folders.
*
* Inherits mailbox validation from the mailbox controller.  Mailbox is a required field for all actions on this controller.
*
* @author           <bell_adam@bah.com>
* @author M. Gibbs <gibbs_margaret@bah.com>
* @author Elan Jaffee <jaffee_elan@bah.com>
*
* @package direct-as-a-service
* @subpackage controllers
*/
class Folder_controller extends Mailbox_controller{	

	/**
	* Returns a list of all folder names for this mailbox, along with the number of new messages and the total number of messages for each folder.
	*
	* Note that in this case, "folders" does *not* correlate to the model concept of folders - it includes all mailbox locations along with the custom folders.
	*
	* This method is available to both active and inactive mailboxes.
	*
	* @param string $mailbox Mailbox name (required)
	*/
	public function folders_get(){		
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('retrieve');
		$this->respond_with_error_if_fields_are_invalid();

		$folders = array( array('id' =>'inbox', 
								'name' => 'Inbox', 
								'total' => $this->mailbox->inbox_message_count(),
								'new'=> $this->mailbox->inbox_message_count(array('seen' => 0))),
						  array('id' =>'sent',
						  		'name' => 'Sent', 
								'total' => $this->mailbox->sent_message_count(),
								'new'=> $this->mailbox->sent_message_count(array('seen' => 0))),
						  array('id' =>'draft',
						  		'name' => 'Drafts', 
								'total' => $this->mailbox->draft_count(),
								'new'=> $this->mailbox->draft_count(array('seen' => 0))),
						  array('id' =>'archived',
						  		'name' => 'Archive', 
								'total' => $this->mailbox->archived_message_count(),
								'new'=> $this->mailbox->archived_message_count(array('seen' => 0))));
						  
		foreach($this->mailbox->folders as $folder){
			$folders[] = array('id' =>$folder->id, 
							   'name' => $folder->name, 
							   'total' => $folder->message_count(),
							   'new'=> $folder->message_count(array('seen' => 0)));
		}
		
#TODO - SHOULDN'T THIS BE PLURAL 'FOLDERS'?  CREATE_FOLDER & RENAME_FOLDER RETURN A SINGLE FOLDER VALUES ARRAY AS 'FOLDER' AND ARCHIVE_FOLDER RETURNS THE FOLDER NAME AS 'FOLDER.'  CONSISTENCY?		
		$this->response_message['folder'] = $folders;
		$this->response($this->response_message, 200);
	}
	
	
	/**
	* Creates a new folder for the given mailbox.
	*
	* This action is not available for inactive mailboxes.
	*
	* @param string $mailbox Mailbox name (required)
	* @param string $name Name for the new folder (required)
	*/
	public function create_folder_post(){		
		$required_fields = array('name');
		$post = $this->post();
        foreach($required_fields as $required_field){
        	if(array_key_exists($required_field, $post))
            	$$required_field = $this->post($required_field,TRUE);
        	else{
        		$this->missing_required_fields[] = $required_field;
           	}
        }
		
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('manage');	
		$this->respond_with_error_if_mailbox_is_inactive();
		
		//make sure that this name is not a reserved location
		if(empty($name) || in_array(mb_strtolower($name), $this->valid_locations())){
			$this->invalid_fields[] = 'name';
		}

		$this->respond_with_error_if_fields_are_invalid();
		
		if($this->mailbox->has_related('folder', compact('name'))){
			$this->response('Folder name already exists for this mailbox.', 422);
		}
		
		$folder = $this->mailbox->add_folder(compact('name'));
		if(!Folder::is_an_entity($folder)){
			$this->response('Failed to create folder.', 400);
		}
		
		$this->response_message['folder'] = $folder->values();
		$this->response($this->response_message, 200);
	}
	
	/**
	* Renames an existing folder to a new name.  The name must be unique for that user.
	*
	* This action is not available for inactive mailboxes.
	*	
	* @param string $mailbox Mailbox name (required)
	* @param int $folder ID of the custom folder (required)
	* @param string $name Name to change the folder to (required)
	*/
	public function rename_folder_post(){		
		$post = $this->post();
		$required_fields = array('name', 'folder');
        foreach($required_fields as $required_field){
        if(array_key_exists($required_field, $post))
            $$required_field = $this->post($required_field,TRUE);
        else{
            $this->missing_required_fields[] = $required_field;
           }
        }
				
		//if there is an error with the passed in fields create an error message
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('manage');				
		$this->respond_with_error_if_mailbox_is_inactive();
		
		$folder_id = $folder; //since the parameter isn't named clearly, we'll set a var that's more intuitive.

		//make sure we're not using a reserved name
		if(empty($name) || in_array(mb_strtolower($name), $this->valid_locations()) ){
			$this->invalid_fields[] = 'name';
		}
		
		//make sure that the folder we're referencing exists		
		if(Folder::formatted_like_an_id($folder_id)) 
			$folder = first_element($this->mailbox->folders( array('id' => $folder_id))) ;
		if(!Folder::is_an_entity($folder) || !$folder->belongs_to_mailbox($this->mailbox)) $this->invalid_fields[] = 'folder';

		$this->respond_with_error_if_fields_are_invalid();
		
		if($this->mailbox->has_related('folder', array( 'name' => $name, 'id !=' => $folder_id)))
			$this->response('Folder name already exists for this user', 422);
		
		$folder->name = $name;
		$folder->save();
		$this->response_message['folder'] = $folder->values();
		$this->response($this->response_message, 200);
	}
	
	/**
	* Archives a custom folder and all the messages within it.
	*
	* This action is not available for inactive mailboxes.
	*
	* @param string $mailbox mailbox name (required)
	* @param int $folder id of the custom folder (required)
	*/
	public function archive_folder_post(){		
		$required_fields = array('folder');
		$post = $this->post();
        foreach($required_fields as $required_field){
        if(array_key_exists($required_field, $post))
            $$required_field = $this->post($required_field,TRUE);
        else{
        	array_push($this->missing_required_fields, $required_field);
           }
        }
		//if there is an error with the passed in fields create an error message
		$this->respond_with_error_if_fields_are_missing();
		$this->respond_with_error_if_user_is_unauthorized('manage');		
		$this->respond_with_error_if_mailbox_is_inactive();

		$folder_id = $folder; //since the parameter isn't named clearly, we'll set a var that's more intuitive.
		
#TODO - SHOULDN'T THIS BE THE FULL FOLDER VALUES ARRAY? CREATE_FOLDER AND RENAME_FOLDER BOTH RETURN THESE AS 'FOLDER.'  OTHERWISE, WE COULD CALL THIS 'FOLDER_NAME' TO BE CLEAR RE:DIFFERENCE.
		$this->response_message['folder'] = $folder_id;
		if(Folder::formatted_like_an_id($folder_id))
			$folder = Folder::find_one($folder_id); 
		if(!Folder::is_an_entity($folder) || !$folder->belongs_to_mailbox($this->mailbox)){
			$this->invalid_fields[] = 'folder';
		}
			
		$this->respond_with_error_if_fields_are_invalid();
		
		$folder->archive();
		$this->response($this->response_message, 200);
	}	
}
?>